
using GeoCrsTransform;
using GeoCrsTransform.MinimalApiSample.Models;

var builder = WebApplication.CreateBuilder(args);
builder.Services.AddEndpointsApiExplorer();
builder.Services.AddSwaggerGen();
builder.Services.AddSingleton<CoordinateTransformer>();

var app = builder.Build();

if (app.Environment.IsDevelopment())
{
    app.UseSwagger();
    app.UseSwaggerUI();
}

app.MapPost("/transform", (TransformRequest req, CoordinateTransformer transformer) =>
{
    try
    {
        var fromCrs = CrsId.Parse(req.From);
        var toCrs = CrsId.Parse(req.To);
        var input = new GeoCoordinate(req.Lat, req.Lon);
        var result = transformer.Transform(input, fromCrs, toCrs);

        if (result is ProjectedCoordinate pc)
        {
            return Results.Ok(new TransformResponse
            {
                Easting = pc.Easting,
                Northing = pc.Northing,
                Crs = toCrs.ToString()
            });
        }
        else if (result is GeoCoordinate gc)
        {
            return Results.Ok(new TransformResponse
            {
                Lat = gc.Latitude,
                Lon = gc.Longitude,
                Crs = toCrs.ToString()
            });
        }
        else
        {
            return Results.BadRequest(new TransformResponse { Error = "Unknown result type." });
        }
    }
    catch (Exception ex)
    {
        return Results.BadRequest(new TransformResponse { Error = ex.Message });
    }
})
.WithName("TransformCoordinate")
.WithOpenApi(op =>
{
    op.Summary = "Transform coordinates between CRSs";
    op.RequestBody = new Microsoft.OpenApi.Models.OpenApiRequestBody
    {
        Content =
        {
            ["application/json"] = new Microsoft.OpenApi.Models.OpenApiMediaType
            {
                Example = new Microsoft.OpenApi.Any.OpenApiObject
                {
                    ["lat"] = new Microsoft.OpenApi.Any.OpenApiDouble(51.5074),
                    ["lon"] = new Microsoft.OpenApi.Any.OpenApiDouble(-0.1278),
                    ["from"] = new Microsoft.OpenApi.Any.OpenApiString("WGS84"),
                    ["to"] = new Microsoft.OpenApi.Any.OpenApiString("OSGB36")
                }
            }
        }
    };
    return op;
});

app.MapGet("/crs", (CoordinateTransformer transformer) =>
{
    var catalog = transformer.Catalog;
    var crsList = catalog.GetAll().Select(c => new { c.Id, c.Name });
    return Results.Ok(crsList);
})
.WithName("GetCrsList")
.WithOpenApi(op =>
{
    op.Summary = "List supported CRSs";
    return op;
});

app.Run();
